///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
// permut to/from ios dof numbering, for distributed environment
#include "Pkd_set_ios_permutations.h"
namespace rheolef {
using namespace std;

// sequential: ios numbering unchanged => nothing to do
template <class T, class M>
void
Pkd_set_ios_permutations (
    const geo_basic<T,M>& omega,
    size_t			       degree,
    array<distributor::size_type,M>&   idof2ios_dis_idof,
    array<distributor::size_type,M>&   ios_idof2dis_idof,
    const boost::array<distributor::size_type,reference_element::max_variant>& loc_ndof_by_variant)
{
}
// distributed: ios numbering is changed
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
void
Pkd_set_ios_permutations (
    const geo_basic<T,distributed>&              omega,
    size_t					 degree,
    array<distributor::size_type,distributed>&   idof2ios_dis_idof,
    array<distributor::size_type,distributed>&   ios_idof2dis_idof,
    const boost::array<distributor::size_type,reference_element::max_variant>& loc_ndof_by_variant)
{
  // general method:
  // - first compute idof2ios_dis_idof
  // - then invert permutation into ios_idof2dis_idof
  typedef distributor::size_type size_type;
  // ----------------------------
  // 1) compute idof2ios_dis_idof
  // ----------------------------
  size_type map_dim = omega.map_dimension();
  const geo_size& gs = omega.sizes(); 
  size_type ndof = 0;
  size_type dis_ndof = 0;
  for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                 variant < reference_element:: last_variant_by_dimension(map_dim);
	         variant++) { 
    ndof     += gs.ownership_by_variant [variant].size()     * loc_ndof_by_variant [variant];
    dis_ndof += gs.ownership_by_variant [variant].dis_size() * loc_ndof_by_variant [variant];
  }
  distributor dof_ownership (dis_ndof, omega.comm(), ndof);
  idof2ios_dis_idof.resize  (dof_ownership);
  switch (map_dim) {
	case 0: 
	case 1: {
	  // non-mixed meshes:
	  size_type loc_ndof_per_elt;
          if (map_dim == 0) {
            loc_ndof_per_elt = loc_ndof_by_variant [reference_element::p];
          } else {
            loc_ndof_per_elt = loc_ndof_by_variant [reference_element::e];
          }
          for (size_type ige = 0, nge = omega.geo_element_ownership(map_dim).size(); ige < nge; ige++) {
	    size_type ios_dis_ige = omega.ige2ios_dis_ige (map_dim, ige);
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_elt; loc_idof++) {
	      size_type idof         = loc_ndof_per_elt*ige         + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_elt*ios_dis_ige + loc_idof;
	      idof2ios_dis_idof [idof] = ios_dis_idof;
            }
          }
	  break;
	}
	case 2: {
	  // possibly (t,q) mixed meshes:
	  // part 1: triangles
          size_type loc_ndof_per_tri = loc_ndof_by_variant [reference_element::t];
          size_type     ntri = omega.sizes().ownership_by_variant[reference_element::t].size();
          size_type dis_ntri = omega.sizes().ownership_by_variant[reference_element::t].dis_size();
          for (size_type itri = 0; itri < ntri; itri++) {
	    size_type ige = itri;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    size_type ios_dis_itri = ios_dis_ige;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tri; loc_idof++) {
	      size_type idof         = loc_ndof_per_tri*itri + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tri*ios_dis_itri + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 2: quadrangle
          size_type loc_ndof_per_qua = loc_ndof_by_variant [reference_element::q];
          size_type nqua = omega.sizes().ownership_by_variant[reference_element::q].size();
          for (size_type iqua = 0; iqua < nqua; iqua++) {
	    size_type ige = ntri + iqua;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntri, "unexpected numbering for element of variant `q'");
	    size_type ios_dis_iqua = ios_dis_ige - dis_ntri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_qua; loc_idof++) {
	      size_type idof = loc_ndof_per_tri*ntri + loc_ndof_per_qua*iqua + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tri*dis_ntri + loc_ndof_per_qua*ios_dis_iqua + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  break;
	}
	case 3:
        default: {
	  // possibly (T,P,H) mixed meshes:
	  // part 1: tetra
          size_type loc_ndof_per_tet = loc_ndof_by_variant [reference_element::T];
          size_type     ntet = omega.sizes().ownership_by_variant[reference_element::T].size();
          size_type dis_ntet = omega.sizes().ownership_by_variant[reference_element::T].dis_size();
          for (size_type itet = 0; itet < ntet; itet++) {
	    size_type ige = itet;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    size_type ios_dis_itet = ios_dis_ige;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tet; loc_idof++) {
	      size_type idof         = loc_ndof_per_tet*itet + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*ios_dis_itet + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 2: prism
          size_type loc_ndof_per_pri = loc_ndof_by_variant [reference_element::P];
          size_type     npri = omega.sizes().ownership_by_variant[reference_element::P].size();
          size_type dis_npri = omega.sizes().ownership_by_variant[reference_element::P].dis_size();
          for (size_type ipri = 0; ipri < npri; ipri++) {
	    size_type ige = ntet + ipri;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntet, "unexpected numbering for element of variant `P'");
	    size_type ios_dis_ipri = ios_dis_ige - dis_ntet;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_pri; loc_idof++) {
	      size_type idof = loc_ndof_per_tet*ntet + loc_ndof_per_pri*ipri + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*dis_ntet + loc_ndof_per_pri*ios_dis_ipri + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 3: hexa
          size_type loc_ndof_per_hex = loc_ndof_by_variant [reference_element::H];
          size_type nhex = omega.sizes().ownership_by_variant[reference_element::H].size();
          for (size_type ihex = 0; ihex < nhex; ihex++) {
	    size_type ige = ntet + npri + ihex;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntet + dis_npri, "unexpected numbering for element of variant `H'");
	    size_type ios_dis_ihex = ios_dis_ige - dis_ntet - dis_npri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_hex; loc_idof++) {
	      size_type idof = loc_ndof_per_tet*ntet + loc_ndof_per_pri*npri + loc_ndof_per_hex*ihex + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*dis_ntet + loc_ndof_per_pri*dis_npri + loc_ndof_per_hex*ios_dis_ihex + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  break;
	}
  }
  // ----------------------------------------------
  // 2) invert permutation into ios_idof2dis_idof
  // ----------------------------------------------
  distributor ios_dof_ownership (dis_ndof, idof2ios_dis_idof.comm(), distributor::decide);
  ios_idof2dis_idof.resize (ios_dof_ownership, std::numeric_limits<size_type>::max());
  idof2ios_dis_idof.reverse_permutation (ios_idof2dis_idof);
}
#endif // _RHEOLEF_HAVE_MPI
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------
#define _RHEOLEF_instanciation(T,M)                                	\
template								\
void Pkd_set_ios_permutations (						\
    const geo_basic<T,M>&,						\
    size_t,								\
    array<distributor::size_type,M>&,					\
    array<distributor::size_type,M>&,					\
    const boost::array<distributor::size_type,reference_element::max_variant>&);

_RHEOLEF_instanciation(Float,sequential)
#ifdef _RHEOLEF_HAVE_MPI
_RHEOLEF_instanciation(Float,distributed)
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
