% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/downsample.R
\name{step_downsample}
\alias{step_downsample}
\alias{tidy.step_downsample}
\title{Down-Sample a Data Set Based on a Factor Variable}
\usage{
step_downsample(
  recipe,
  ...,
  under_ratio = 1,
  ratio = deprecated(),
  role = NA,
  trained = FALSE,
  column = NULL,
  target = NA,
  skip = TRUE,
  seed = sample.int(10^5, 1),
  id = rand_id("downsample")
)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the
sequence of operations for this recipe.}

\item{...}{One or more selector functions to choose which
variable is used to sample the data. See \link[recipes:selections]{recipes::selections}
for more details. The selection should result in \emph{single
factor variable}. For the \code{tidy} method, these are not
currently used.}

\item{under_ratio}{A numeric value for the ratio of the
majority-to-minority frequencies. The default value (1) means
that all other levels are sampled down to have the same
frequency as the least occurring level. A value of 2 would mean
that the majority levels will have (at most) (approximately)
twice as many rows than the minority level.}

\item{ratio}{Deprecated argument; same as \code{under_ratio}}

\item{role}{Not used by this step since no new variables are
created.}

\item{trained}{A logical to indicate if the quantities for
preprocessing have been estimated.}

\item{column}{A character string of the variable name that will
be populated (eventually) by the \code{...} selectors.}

\item{target}{An integer that will be used to subsample. This
should not be set by the user and will be populated by \code{prep}.}

\item{skip}{A logical. Should the step be skipped when the
recipe is baked by \code{\link[recipes:bake]{bake()}}? While all operations are baked
when \code{\link[recipes:prep]{prep()}} is run, some operations may not be able to be
conducted on new data (e.g. processing the outcome variable(s)).
Care should be taken when using \code{skip = TRUE} as it may affect
the computations for subsequent operations.}

\item{seed}{An integer that will be used as the seed when downsampling.}

\item{id}{A character string that is unique to this step to identify it.}
}
\value{
An updated version of \code{recipe} with the new step
added to the sequence of existing steps (if any). For the
\code{tidy} method, a tibble with columns \code{terms} which is
the variable used to sample.
}
\description{
\code{step_downsample()} creates a \emph{specification} of a recipe step that will
remove rows of a data set to make the occurrence of levels in a specific
factor level equal.
}
\details{
Down-sampling is intended to be performed on the \emph{training} set
alone. For this reason, the default is \code{skip = TRUE}.

If there are missing values in the factor variable that is used
to define the sampling, missing data are selected at random in
the same way that the other factor levels are sampled. Missing
values are not used to determine the amount of data in the
minority level

For any data with factor levels occurring with the same
frequency as the minority level, all data will be retained.

All columns in the data are sampled and returned by \code{\link[recipes:juice]{recipes::juice()}}
and \code{\link[recipes:bake]{recipes::bake()}}.

Keep in mind that the location of down-sampling in the step
may have effects. For example, if centering and scaling,
it is not clear whether those operations should be conducted
\emph{before} or \emph{after} rows are removed.
}
\section{Tidying}{
When you \code{\link[recipes:tidy.recipe]{tidy()}} this step, a tibble is retruned with
columns \code{terms} and \code{id}:

\describe{
\item{terms}{character, the selectors or variables selected}
\item{id}{character, id of this step}
}
}

\section{Tuning Parameters}{
This step has 1 tuning parameters:
\itemize{
\item \code{under_ratio}: Under-Sampling Ratio (type: double, default: 1)
}
}

\section{Case weights}{


This step performs an unsupervised operation that can utilize case weights.
To use them, see the documentation in \link[recipes:case_weights]{recipes::case_weights} and the examples on
\code{tidymodels.org}.
}

\examples{
\dontshow{if (rlang::is_installed("modeldata")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(recipes)
library(modeldata)
data(hpc_data)

hpc_data0 <- hpc_data \%>\%
  select(-protocol, -day)

orig <- count(hpc_data0, class, name = "orig")
orig

up_rec <- recipe(class ~ ., data = hpc_data0) \%>\%
  # Bring the majority levels down to about 1000 each
  # 1000/259 is approx 3.862
  step_downsample(class, under_ratio = 3.862) \%>\%
  prep()

training <- up_rec \%>\%
  bake(new_data = NULL) \%>\%
  count(class, name = "training")
training

# Since `skip` defaults to TRUE, baking the step has no effect
baked <- up_rec \%>\%
  bake(new_data = hpc_data0) \%>\%
  count(class, name = "baked")
baked

# Note that if the original data contained more rows than the
# target n (= ratio * majority_n), the data are left alone:
orig \%>\%
  left_join(training, by = "class") \%>\%
  left_join(baked, by = "class")

library(ggplot2)

ggplot(circle_example, aes(x, y, color = class)) +
  geom_point() +
  labs(title = "Without downsample")

recipe(class ~ x + y, data = circle_example) \%>\%
  step_downsample(class) \%>\%
  prep() \%>\%
  bake(new_data = NULL) \%>\%
  ggplot(aes(x, y, color = class)) +
  geom_point() +
  labs(title = "With downsample")
\dontshow{\}) # examplesIf}
}
\seealso{
Other Steps for under-sampling: 
\code{\link{step_nearmiss}()},
\code{\link{step_tomek}()}
}
\concept{Steps for under-sampling}
