////////////////////////////////////////////////////////////////
//
// Copyright (C) 2005 Affymetrix, Inc.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License 
// (version 2.1) as published by the Free Software Foundation.
// 
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
// for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this library; if not, write to the Free Software Foundation, Inc.,
// 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
//
////////////////////////////////////////////////////////////////

#ifndef _DttArrayFileReader_HEADER_
#define _DttArrayFileReader_HEADER_

/*! \file DttArrayFileReader.h This file provides interfaces to read an array file generated by the DTT1.1 software.
 *
 * The Dtt array file is the MAGE-ML file generated by the DTT 1.1 software.
 */

#include <cstring>
#include <list>
#include <string>
//

namespace affymetrix_dttarray
{

/*! The parameter name for the gcos sample name. */
#define GCOS_SAMPLE_NAME_PARAMETER_NAME "GCOS Sample Name"

/*! The parameter name for the gcos sample type. */
#define GCOS_SAMPLE_TYPE_PARAMETER_NAME "GCOS Sample Type"

/*! The parameter name for the gcos sample project. */
#define GCOS_SAMPLE_PROJECT_PARAMETER_NAME "GCOS Sample Project"

/*! A structure to hold an attribute name/value/type. */
typedef struct _AttributeNameValueType
{
	/*! The attribute name. */
	std::string name;

	/*! The attribute value. */
	std::string value;

	/*! The attribute type. */
	std::string type;

} AttributeNameValueType;

/*! An STL list of attributes. */
typedef std::list<AttributeNameValueType> AttributeNameValueTypeList;

/*! This class provides storage of the DTT array file data. */
class DttArrayData
{
private:
	/*! The probe array type. */
	std::string arrayType;

	/*! The experiment name. */
	std::string expName;

	/*! The attributes. */
	AttributeNameValueTypeList attributes;

public:
	/*! Constructor */
	DttArrayData();

	/*! Destructor */
	~DttArrayData();

	/*! Gets the probe array type name.
	 * @return The probe array type.
	 */
	std::string GetArrayType() const { return arrayType; }

	/*! Sets the probe array type name.
	 * @param aType The array type.
	 */
	void SetArrayType(std::string const &aType) { arrayType = aType; }

	/*! Gets the experiemnt name.
	 * @return The experiment name.
	 */
	std::string GetExperimentName() const { return expName; }

	/*! Sets the experiment name.
	 * @param name The experiment name.
	 */
	void SetExperimentName(std::string const &name) { expName = name; }

	/*! Gets the attributes.
	 * @return The sample and experiemnt attributes.
	 */
	AttributeNameValueTypeList &Attributes() { return attributes; }

	/*! Clears the members. */
	void Clear();
};

/*! This class provides interfaces to read an array file. */
class DttArrayFileReader
{
protected:
	/*! The full path name to the DTT array file. */
	std::string m_FileName;

public:
	/*! Constructor */
	DttArrayFileReader();

	/*! Destructor */
	~DttArrayFileReader();

	/*! Gets the file name.
	 * @return The full path of the array file.
	 */
	std::string GetFileName() const { return m_FileName; }

	/*! Sets the file name.
	 * @param file The full path of the array file.
	 */
	void SetFileName(const char *file) { m_FileName = file; }

	/*! Reads the entire contents of the array file.
	 * @param arrayData The array data to read from the file.
	 * @return True if successfully read.
	 */
	bool Read(DttArrayData &arrayData);

	/*! Checks for the existance of the file.
	 * @return True if the file exists.
	 */
	bool Exists();
};

};

#endif // _DttArrayFileReader_HEADER_
