function fig = EdulogPlot(data, loggers)
% Create a plot from Edulog data
%
% "data" is a structure generated by running an Edulogger experiment,
% consisting of the following fields: Time: The time (s) since the start of
% the experiment of each sample. (double) Concern: Whether or not each
% sample took more than twice the specified sample rate to retrieve
% (logical) Event (optional): Whether or not an event happened at this
% point (logical) An additional field for each kind of Edulogger used,
% containing the measurements taken at each point in data.Time. Fieldnames
% should line up with the names specified in "loggers". "loggers" is a one
% dimensional cell array, with each string specifying the name of a
% different Edulogger as described in the Neulog API literature:
% https://neulog.com/wp-content/uploads/2014/06/NeuLog-API-version-7.pdf
%
% "fig" is a Graphics Object containing the graph generated, properties of
% the graph can be changed by editing this object.

% History:
% ??-??-????    Todd Parsons    Written.

sDim = get(0,'screensize'); % Get screensize

close all; % Close any open figures

% Create & setup a blank figure
fig = figure; % Create figure
set(fig, 'Name', 'Edulog Data'); % Name figure
set(fig, 'NumberTitle', 'off'); % Remove "Figure 1" label
set(fig, 'Color', 'white'); % White background
pos = [200, 100, sDim(3) - 400, sDim(4) - 200];
set(fig, 'Position', pos); % Resize to the height of the screen - 200

% Extract data
m = min(length(loggers), 3); % Determine number of rows (max 3)
n = ceil(length(loggers)/3); % Determine number of columns
e = [data.Event]; % Find events
c = [data.Concern]; % Find latency points
x = [data.Time]; % Extract x data

for L = 1:length(loggers)
    % Get data
    y = [data.(loggers{L})];

    % Setup axis
    ax{L} = subplot(m, n, L); % Choose sub-plot to draw in
    rect = get(ax{L}, 'Position');
    rect([1,3]) = [0.1, 0.8];
    set(ax{L}, 'Position', rect); % Position plot
    set(ax{L}, 'FontName', 'Verdana'); % Change font
    set(get(ax{L}, 'XLabel'), 'String', 'Time (s)'); % Label x-axis
    set(get(ax{L}, 'YLabel'), 'String', loggers{L}); % Label y-axis
    set(ax{L}, 'Color', [0.98, 0.98, 1]); % Axis background
    set(ax{L}, 'XGrid', 'on'); % Add vertical gridlines
    set(ax{L}, 'YGrid', 'on'); % Add horizontal gridlines
    set(ax{L}, 'GridColor', 'white'); % Make gridlines white
    set(ax{L}, 'GridAlpha', 1); % Make gridlines opaque

    % Plot data
    ln{L} = line(x, y, 'Color', [42, 107, 211]./255, 'LineWidth', 2); %#ok<*NASGU,*AGROW> % Plot data
    ev{L} = line([[data(e).Time]; [data(e).Time]]', get(ax{L}, 'YLim'), 'Color', 'k', 'LineWidth', 2); % Plot events
    co{L} = line([[data(c).Time]; [data(c).Time]]', get(ax{L}, 'YLim'), 'Color', 'r', 'LineStyle', ':'); % Plot concern points

    % Draw legend
    le{L} = legend({"Data", "Events", "Concern"}); %#ok<CLARRSTR> % Add legend
    rect = get(le{L}, 'Position');
    rect([1,3]) = [0.9, 0.1];
    set(le{L}, 'Position', rect); % Position legend
    set(le{L}, 'Box', 'off'); % Remove outline
end
