/*
 * radclient  A client-side implementation of the RADIUS protocol.
 *
 * Version:  @(#)radclient.c  1.43  08-Nov-1997  miquels@cistron.nl
 *
 */
#include <arpa/inet.h>
#include <errno.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <time.h>
#include <unistd.h>
#include <signal.h>

#ifndef NO_CHAP
#include <chap.h>
#ifdef CHAPMS
#include <chap_ms.h>
#endif
#endif

#include "server.h"
#include <radiusclient.h>

extern void radius_md5_calc(unsigned char *, unsigned char *, unsigned int);

/*
 *  This is an "attribute".
 */
typedef struct _attr_
{
  unsigned char type;
  unsigned char len;
  union {
    char str[254];
    unsigned int num;
  } val;
  unsigned char pwtype;
  struct _attr_ *next;
} ATTR;

static int update_framed_route(const struct auth *ai, bool islogin);
static int update_filter_id(const struct auth *ai, bool islogin);

/* Add the basic data to an accounting or authorization packet */
static bool rad_add_base_data(VALUE_PAIR *pkt, const struct auth *ai)
{
  /*
   *  Now we finish off with the client id (our ip address),
   */
  rc_avpair_add(&pkt, PW_NAS_PORT_TYPE, (void *)&ai->porttype, 0);
  /*
   *  Add connection info on login AND logout. That's what
   *  a portmaster does too..
   */
  if(ai->conn_info[0])
    rc_avpair_add(&pkt, PW_CONNECT_INFO, (void *)ai->conn_info, 0);
  if(ai->cli_src[0])
    rc_avpair_add(&pkt, PW_CALLING_STATION_ID, (void *)ai->cli_src, 0);
  if(ai->cli_dst[0])
    rc_avpair_add(&pkt, PW_CALLED_STATION_ID, (void *)ai->cli_dst, 0);

  /*
   *  We have to add a unique identifier, the same
   *  for login as for logout.
   */
  rc_avpair_add(&pkt, PW_ACCT_SESSION_ID, ai->acct_session_id, 0);

  return 0;
}

/*
 *  Build an authentication request.
 */
static VALUE_PAIR *rad_buildauth(const struct auth *ai, int ppp)
{
  VALUE_PAIR *pkt;
  UINT4 av_type;

  pkt = NULL;
  /*
   *  Add the login name.
   */
  rc_avpair_add(&pkt, PW_USER_NAME, (void *)ai->login, 0);

  if(ai->called_station)
  {
    rc_avpair_add(&pkt, PW_CALLED_STATION_ID, ai->called_station, 0);
  }
  if(ai->calling_station)
  {
    rc_avpair_add(&pkt, PW_CALLING_STATION_ID, ai->calling_station, 0);
  }

  if(rad_add_base_data(pkt, ai))
    return NULL;


  /*
   *  We add the protocol type if this is PPP.
   */
  if(ppp)
  {
    av_type = htonl(PW_PPP);
    rc_avpair_add(&pkt, PW_FRAMED_PROTOCOL, &av_type, 0);
    av_type = htonl(PW_FRAMED);
    rc_avpair_add(&pkt, PW_SERVICE_TYPE, &av_type, 0);
  }

  return pkt;
}

/*
 *  Build an accounting request.
 */
static VALUE_PAIR *rad_buildacct(const struct auth *ai, bool islogin)
{
  int i, s, p, c;
  unsigned int h;
  VALUE_PAIR *pkt = NULL;
  UINT4 av_type;

  /*
   *  Tell the server what kind of request this is.
   */
  av_type = islogin ? PW_STATUS_START : PW_STATUS_STOP;
  rc_avpair_add(&pkt, PW_ACCT_STATUS_TYPE, &av_type, 0);

  /*
   *  Add the login name.
   */
  if(ai->login && ai->login[0])
    rc_avpair_add(&pkt, PW_USER_NAME, (void *)ai->login, 0);

  if(rad_add_base_data(pkt, ai))
    return NULL;

  if(!islogin)
  {
    /*
     *  Add input and output octets.
     */
    if(ai->traffic.sent_bytes || ai->traffic.recv_bytes)
    {
      rc_avpair_add(&pkt, PW_ACCT_OUTPUT_OCTETS, (void *)&ai->traffic.sent_bytes, 0);
      rc_avpair_add(&pkt, PW_ACCT_INPUT_OCTETS, (void *)&ai->traffic.recv_bytes, 0);
    }
    /*
     *  Add input and output packets.
     */
    if(ai->traffic.sent_pkts || ai->traffic.recv_pkts)
    {
      rc_avpair_add(&pkt, PW_ACCT_OUTPUT_PACKETS, (void *)&ai->traffic.sent_pkts, 0);
      rc_avpair_add(&pkt, PW_ACCT_INPUT_PACKETS, (void *)&ai->traffic.recv_pkts, 0);
    }
  }

  /*
   *  Include session time if this is a logout.
   */
  if(!islogin)
  {
    av_type = time(NULL) - ai->start;
    rc_avpair_add(&pkt, PW_ACCT_SESSION_TIME, &av_type, 0);
  }

  /*
   *  We'll have to add some more fields here:
   *  - service type
   *  - login service
   *  - framed protocol
   *  - framed IP address
   *  - framed compression
   *  - login IP host
   *
   */
  i = -1;
  s = -1;
  p = -1;
  c = -1;
  h = 0;
  switch(ai->proto)
  {
    case P_SHELL:
      s = PW_ADMINISTRATIVE;
    break;
    case P_TELNET:
      s = PW_LOGIN;
      i = PW_TELNET;
      h = ai->address;
    break;
    case P_RLOGIN:
      s = PW_LOGIN;
      i = PW_RLOGIN;
      h = ai->address;
    break;
    case P_SSH1:
    case P_SSH2:
      s = PW_LOGIN;
      i = PW_SSH;
      h = ai->address;
    break;
    case P_TCPCLEAR:
    case P_TCPLOGIN:
      s = PW_LOGIN;
      i = PW_TCP_CLEAR;
      h = ai->address;
    break;
    case P_PPP:
    case P_PPP_ONLY:
    case P_SLIP:
    case P_CSLIP:
    {
      unsigned long address = htonl(ai->address);
      s = PW_FRAMED;
      rc_avpair_add(&pkt, PW_FRAMED_IP_ADDRESS, &address, 0);
    }
    break;
  }
  switch(ai->proto)
  {
    case P_PPP:
    case P_PPP_ONLY:
      p = PW_PPP;
      c = PW_VAN_JACOBSON_TCP_IP;
    break;
    case P_SLIP:
      p = PW_SLIP;
    break;
    case P_CSLIP:
      p = PW_SLIP;
      c = PW_VAN_JACOBSON_TCP_IP;
  }
  if(s > 0)
  {
    av_type = s;
    rc_avpair_add(&pkt, PW_SERVICE_TYPE, &av_type, 0);
  }
  if(i >= 0)
  {
    av_type = i;
    rc_avpair_add(&pkt, PW_LOGIN_SERVICE, &av_type, 0);
  }
  if(p >= 0)
  {
    av_type = p;
    rc_avpair_add(&pkt, PW_FRAMED_PROTOCOL, &av_type, 0);
  }
  if(c >= 0)
  {
    av_type = c;
    rc_avpair_add(&pkt, PW_FRAMED_COMPRESSION, &av_type, 0);
  }
  if(h > 0)
  {
    rc_avpair_add(&pkt, PW_LOGIN_IP_HOST, &h, 0);
  }

  return pkt;
}

static void free_arrray(char **messages, int num)
{
  int i;
  for(i = 0; i < num; i++)
  {
    if(messages[i])
      free(messages[i]);
    messages[i] = NULL;
  }
}

/*
 *  set environment variable name with a representation of the array val of
 *  the number of strings represented by num
 */
int setenv_from_rad(const char *name, const char **val, unsigned int num)
{
  unsigned int i;
  int size = 0;
  char *buf;
  int rc = 0;

  if(num == 0)
    return 0;

  for(i = 0; i < num; i++)
    size += strlen(val[i]);

  buf = xmalloc(size + num);
  for(i = 0; i < num; i++)
  {
    strcat(buf, val[i]);
    if(i != num - 1)
      strcat(buf, "#");
  }
  if(setenv(name, buf, 1))
  {
    nsyslog(LOG_ERR, "Can't set environment variable %s.", name);
    rc = -1;
  }

  free(buf);
  return rc;
}

/*
 *  get data from environment variable name and put it in the array val of
 *  the maximum number of strings represented by max, too many strings cause
 *  an error.  Puts the number of strings in num.
 */
int getenv_from_rad(const char *name, char **val, unsigned int max, unsigned int *num)
{
  char *buf, *ptr, *next;

  *num = 0;
  buf = getenv(name);
  if(!buf)
    return 0;
  ptr = buf;
  next = buf;
  while(next)
  {
    if(*num >= max)
    {
      nsyslog(LOG_ERR, "Error extracting variable %s.", name);
      return -1;
    }
    next = strchr(ptr, '#');
    if(!next)
    {
      val[*num] = xstrdup(ptr);
    }
    else
    {
      val[*num] = xmalloc(next - ptr + 1);
      memcpy(val[*num], ptr, next - ptr);
      val[*num][next - ptr] = '\0';
    }
    (*num)++;
    ptr = next;
  }
  return 0;
}

void unpack_radius_auth_reply(const VALUE_PAIR * const received, struct auth *ai);

/*
 *  Ask a radius server to authenticate us,
 *
 *      ppp parameter is whether the protocol is PPP.
 *      for adding PPP attribute to packet
 */
int rad_client(struct auth *ai, bool ppp)
{
  int ret;
  VALUE_PAIR *pkt, *received;
  int def_auth_port;
  struct servent *svp;
  int result;

  if((ai->passwd[0] == '\0') && (lineconf.radnullpass == 0))
    return (-1);

  /* Get radauth port, or use default */
  svp = getservbyname ("radius", "udp");
  if(svp == NULL)
    def_auth_port = PW_AUTH_UDP_PORT;
  else
    def_auth_port = ntohs(svp->s_port);

  /*
   *  Set the message to some error in case we fail.
   */
  if(ai->message[0])
    free(ai->message[0]);
  ai->message[0] = xmalloc(4096);
  ai->msn = 1;

        /*
         *      First, build the request.
         */
  if ((pkt = rad_buildauth(ai, ppp)) == NULL)
    return -1;

  if(lineconf.logpassword)
    nsyslog(LOG_DEBUG, "passwd: %s", ai->passwd);

  rc_avpair_add(&pkt, PW_USER_PASSWORD, (void *)ai->passwd, 0);

  result = rc_auth(GetPortNo(), pkt, &received, ai->message[0]);
  if(!ai->message[0][0])
  {
    free(ai->message[0]);
    ai->message[0] = NULL;
  }

  /* free value pairs */
  rc_avpair_free(pkt);

  ret = (result == OK_RC) ? 0 : -1;
  free_arrray(ai->message, MAX_RADIUS_MESSAGES);
  ai->msn = 0;
  free_arrray(ai->filterid, MAX_FILTERS);
  ai->fln = 0;

  unpack_radius_auth_reply(received, ai);

  if(ret == 0)
  {
    ai->start = time(NULL);
  }
  else
  {
    nsyslog(LOG_INFO, "authentication failed (%s/%s) %s",
      ai->login, ai->passwd, ai->message[0] ? ai->message[0] : "");
  }
  /* free value pairs */
  rc_avpair_free(received);
  return ret;
}

#ifndef NO_CHAP
int rad_client_chap(struct auth *ai, u_char *remmd, chap_state *cstate)
{
  int ret = CHAP_SUCCESS;
  VALUE_PAIR *pkt, *received;
  int def_auth_port;
  struct servent *svp;
  int result;
  char cpassword[MAX_RESPONSE_LENGTH + 1];

/*  if((ai->passwd[0] == '\0') && (lineconf.radnullpass == 0))
    return CHAP_FAILURE; */

  /* Get radauth port, or use default */
  svp = getservbyname ("radius", "udp");
  if(svp == NULL)
    def_auth_port = PW_AUTH_UDP_PORT;
  else
    def_auth_port = ntohs(svp->s_port);

  /*
   *  Set the message to some error in case we fail.
   */
  if(ai->message[0])
    free(ai->message[0]);
  ai->message[0] = xmalloc(4096);
  ai->msn = 1;

  /*
   *      First, build the request.
   */
  if ((pkt = rad_buildauth(ai, 1)) == NULL)
    return CHAP_FAILURE;

  /* for PAP we optionally log the password here, for CHAP we can't */

  /*
   * add the challenge and response fields
   */
  switch (cstate->chal_type)
  {
    case CHAP_DIGEST_MD5:
      /* CHAP-Challenge and CHAP-Password */
      cpassword[0] = cstate->chal_id;
      memcpy(&cpassword[1], remmd, MD5_SIGNATURE_SIZE);

      rc_avpair_add(&pkt, PW_CHAP_CHALLENGE,
                    cstate->challenge, cstate->chal_len);
      rc_avpair_add(&pkt, PW_CHAP_PASSWORD,
                    cpassword, MD5_SIGNATURE_SIZE + 1);
    break;
#ifdef CHAPMS
    case CHAP_MICROSOFT:
    {
      /* MS-CHAP-Challenge and MS-CHAP-Response */
      MS_ChapResponse *rmd = (MS_ChapResponse *) remmd;
      u_char *p = cpassword;

      *p++ = cstate->chal_id;
      /* The idiots use a different field order in RADIUS than PPP */
      memcpy(p, rmd->UseNT, sizeof(rmd->UseNT));
      p += sizeof(rmd->UseNT);
      memcpy(p, rmd->LANManResp, sizeof(rmd->LANManResp));
      p += sizeof(rmd->LANManResp);
      memcpy(p, rmd->NTResp, sizeof(rmd->NTResp));

      rc_avpair_add(&send, PW_MS_CHAP_CHALLENGE,
                    cstate->challenge, cstate->chal_len, VENDOR_MICROSOFT);
      rc_avpair_add(&send, PW_MS_CHAP_RESPONSE,
                    cpassword, MS_CHAP_RESPONSE_LEN + 1, VENDOR_MICROSOFT);
    }
    break;
    case CHAP_MICROSOFT_V2:
    {
      /* MS-CHAP-Challenge and MS-CHAP2-Response */
      MS_Chap2Response *rmd = (MS_Chap2Response *) remmd;
      u_char *p = cpassword;

      *p++ = cstate->chal_id;
      /* The idiots use a different field order in RADIUS than PPP */
      memcpy(p, rmd->Flags, sizeof(rmd->Flags));
      p += sizeof(rmd->Flags);
      memcpy(p, rmd->PeerChallenge, sizeof(rmd->PeerChallenge));
      p += sizeof(rmd->PeerChallenge);
      memcpy(p, rmd->Reserved, sizeof(rmd->Reserved));
      p += sizeof(rmd->Reserved);
      memcpy(p, rmd->NTResp, sizeof(rmd->NTResp));

      rc_avpair_add(&send, PW_MS_CHAP_CHALLENGE,
                    cstate->challenge, cstate->chal_len, VENDOR_MICROSOFT);
      rc_avpair_add(&send, PW_MS_CHAP2_RESPONSE,
                    cpassword, MS_CHAP2_RESPONSE_LEN + 1, VENDOR_MICROSOFT);
    }
    break;
#endif
  }


  result = rc_auth(GetPortNo(), pkt, &received, ai->message[0]);
  if(!ai->message[0][0])
  {
    free(ai->message[0]);
    ai->message[0] = NULL;
  }

  /* free value pairs */
  rc_avpair_free(pkt);

  if(result != OK_RC)
    ret = CHAP_FAILURE;
  free_arrray(ai->message, MAX_RADIUS_MESSAGES);
  ai->msn = 0;
  free_arrray(ai->filterid, MAX_FILTERS);
  ai->fln = 0;
  if(!ai->done_chap_once && ret == CHAP_SUCCESS)
  {
    unpack_radius_auth_reply(received, ai);
    ai->start = time(NULL);
  }
  else
  {
    nsyslog(LOG_INFO, "authentication failed (%s/CHAP) %s",
      ai->login, ai->message[0] ? ai->message[0] : "");

  }
  /* free value pairs */
  rc_avpair_free(received);
  return ret;
}
#endif

void unpack_radius_auth_reply(const VALUE_PAIR * const received, struct auth *ai)
{
  bool islogin = false;
  bool isframed = false;
  const VALUE_PAIR *ptr;
  int oldproto = ai->proto;

  ai->proto = 0;

  /*
   *  Put the reply into our auth struct.
   */
  for(ptr = received; ptr; ptr = ptr->next)
  {
/*    if (ptr->type == 0) break; */
    switch(ptr->attribute)
    {
      case PW_SERVICE_TYPE:
        /* Framed or login. */
        switch(ptr->lvalue)
        {
          case PW_ADMINISTRATIVE:
            ai->proto = P_SHELL;
          break;
          case PW_LOGIN:
            islogin = true;
          break;
          case PW_FRAMED:
            isframed = true;
          break;
          default:
          break;
        }
      break;
      case PW_LOGIN_SERVICE:
        islogin = true;
        switch(ptr->lvalue)
        {
          case PW_TELNET:
            ai->proto = P_TELNET;
          break;
          case PW_RLOGIN:
            ai->proto = P_RLOGIN;
          break;
          case PW_SSH:
            ai->proto = P_SSH1;
          break;
          case PW_TCP_CLEAR:
            ai->proto = P_TCPCLEAR;
          break;
          case PW_PORTMASTER:
          default:
            islogin = false;
            /* Unknown to us. */
          break;
        }
      break;
      case PW_LOGIN_IP_HOST:
        ai->host = ntohl(ptr->lvalue);
      break;
#ifdef HAVE_IPV6
      case PW_LOGIN_IPV6_HOST:
        ai->host = ntohl(ptr->lvalue);
      break;
#endif
      case PW_LOGIN_PORT:
        ai->loginport = ptr->lvalue;
      break;
      case PW_PORT_LIMIT:
        ai->port_limit = ptr->lvalue;
      break;
      case PW_FRAMED_IP_ADDRESS:
        isframed = true;
        if((unsigned)ntohl(ptr->lvalue) != 0xFFFFFFFE)
          ai->address = ntohl(ptr->lvalue);
      break;
      case PW_FRAMED_IP_NETMASK:
        ai->netmask = ntohl(ptr->lvalue);
      break;
      case PW_FRAMED_MTU:
        ai->mtu = ptr->lvalue;
      break;
      case PW_IDLE_TIMEOUT:
        ai->idletime = ptr->lvalue;
      break;
      case PW_SESSION_TIMEOUT:
        ai->sessiontime = ptr->lvalue;
      break;
      case PW_FRAMED_COMPRESSION:
        if(ptr->lvalue != PW_VAN_JACOBSON_TCP_IP)
          break;
        if (ai->proto == 0 || ai->proto == P_SLIP)
          ai->proto = P_CSLIP;
      break;
      case PW_FRAMED_PROTOCOL:
        isframed = true;
        if(ptr->lvalue == PW_PPP)
          ai->proto = P_PPP;
        else if (ai->proto == 0)
          ai->proto = P_SLIP;
      break;
      case PW_FILTER_ID:
        if(ai->fln > MAX_FILTERS)
          break;
        ai->filterid[ai->fln] = xstrdup(ptr->strvalue);
        ai->fln++;
      break;
      case PW_FRAMED_ROUTE:
        if(ai->frn >= MAX_FRAMED_ROUTES)
          break;
        ai->framed_route[ai->frn] = xstrdup(ptr->strvalue);
        ai->frn++;
      break;
      case PW_REPLY_MESSAGE:
        if(ai->msn >= MAX_RADIUS_MESSAGES)
          break;
        ai->message[ai->msn] = xstrdup(ptr->strvalue);
        ai->msn++;
      break;
#ifdef PORTSLAVE_CALLBACK
      case PW_CALLBACK_NUMBER:
        ai->cb_allowed = (1 << CB_CONF_ADMIN);
        ai->cb_number = xmalloc(len + 1);
        memcpy(ai->cb_number,a_val,len);
        ai->cb_number[len] = 0;
      break;
      case PW_VENDOR_SPECIFIC:
        len = a_len;
        if(len > 6)
        {
          unsigned long lvalue;
          unsigned long vid;
          char *attr;
          memcpy(&lvalue, a_val, 4);
          vid = ntohl(lvalue);
          if(vid == 9)
          { /* Cisco */
            if(((int)a_val[4] == 1) && ((int)a_val[5] >= 24) )
            {
              unsigned int vlen=(int)a_val[5];
              if(strncmp(&a_val[6],"lcp:callback-dialstring",24) == 0)
              {
                if(vlen==24)
                { /* empty string, user defined */
                  ai->cb_allowed = (1 << CB_CONF_USER);
                }
                else
                { /* admin defined number */
                  ai->cb_allowed = (1 << CB_CONF_ADMIN);
                  ai->cb_number = xmalloc(vlen-24+1);
                  memcpy(ai->cb_number,&a_val[30],vlen-24);
                  ai->cb_number[vlen-24]=0;
                }
              }
            }
          }
        }
      break;
#endif
    }
  }

  if(isframed && ai->address == 0 && lineconf.rem_host)
    ai->address = lineconf.rem_host;

  if(islogin && ai->address == 0 && lineconf.host)
    ai->address = lineconf.host;

  if(ai->proto == 0)
    ai->proto = oldproto;
}

/*
 *  Send accounting info to a RADIUS server.
 */
static int real_rad_acct(const struct auth *ai, bool islogin)
{
  VALUE_PAIR *pkt;
  int result;

  /*
   *  Update utmp/wtmp (ctlportslave)
   */
  update_utmp(lineconf.stripnames ? "%L" : "%l", lineconf.utmpfrom, ai, lineconf.syswtmp);

  /*
   *  Update Framed-Routes
   */
  update_framed_route(ai, islogin);

  /*
   *  Run filter backend
   */
  update_filter_id(ai, islogin);

        /*
         *      First, build the request.
         */
  if((pkt = rad_buildacct(ai, islogin)) == NULL)
    return -1;

  result = rc_acct(GetPortNo(), pkt);
  if(result != OK_RC)
  {
    /* RADIUS server could be down so make this a warning */
    syslog(LOG_WARNING, "Accounting STOP failed for %s", ai->login);
  }
  rc_avpair_free(pkt);

  return 0;
}

int rad_acct(const struct auth *ai, bool islogin)
{
  int rc;

  if(!ai->do_acct)
    return 0;

   /*
    *      While messing around with accounting, we have to
    *      block, but not ignore, SIGHUP and friends.
    */
  block(SIGHUP);
  block(SIGTERM);
  rc = real_rad_acct(ai, islogin);
  unblock(SIGTERM);
  unblock(SIGHUP);
  return rc;
}

/*
 * Invoke 'route' command to update "Framed-Route".
 */
static int update_framed_route(const struct auth *ai, bool islogin)
{
  char cmd[1024], *buf = NULL;
  char *net, *gw, *metric;
  const char *net_or_host;
  int x, rc;

  if(ai->frn == 0) return 0;

  if(islogin)
  {
    nsyslog(LOG_INFO, "Adding routes: %d.", ai->frn);
    x = 0;
  }
  else
  {
    nsyslog(LOG_INFO, "Deleting routes: %d.", ai->frn);
    x = ai->frn - 1;
  }

  while(x < (int)ai->frn && x >= 0)
  {
    if(buf) free(buf);
    buf = xstrdup(ai->framed_route[x]);
    net = strtok (buf, " ");
    if(strlen(net) > 3 && !strcmp("/32", &net[strlen(net) - 3]))
    {
      net_or_host = "-host";
      net[strlen(net) - 3] = '\0';
    }
    else
    {
      net_or_host = "-net";
    }
    gw = strtok ((char *) 0, " ");
    if(!gw || !strcmp(gw, "0.0.0.0"))
      gw = xstrdup(dotted(ai->address));
    else
      gw = xstrdup(gw);
    metric = strtok ((char *) 0, " ");

    if(metric)
    {
      snprintf(cmd, sizeof(cmd) - 1, "exec %s %s %s %s gw %s metric %s >/dev/null 2>&1",
      PATH_ROUTE, islogin ? "add" : "del", net_or_host, net, gw, metric);
    }
    else
    {
      snprintf(cmd, sizeof(cmd) - 1, "exec %s %s %s %s gw %s >/dev/null 2>&1",
      PATH_ROUTE, islogin ? "add" : "del", net_or_host, net, gw);
    }
    free(gw);
    rc = system(cmd);
    if(rc)
      nsyslog(LOG_ERR, "Command \"%s\" returned %d", cmd, rc);

    islogin ? x++ : x--;    /* FIFO add, LIFO del */
  }

  if(buf) free(buf);
  return 0;
}

static void alrm_hand()
{
}

/*
 * Invoke the filter script defined by "Framed-Filter-Id".
 */
static int update_filter_id(const struct auth *ai, bool islogin)
{
  int x;

  if(ai->fln == 0) return 0;

  if(islogin)
  {
    nsyslog(LOG_INFO, "Starting filters: %d.", ai->fln);
    x = 0;
  }
  else
  {
    nsyslog(LOG_INFO, "Stopping filters: %d.", ai->fln);
    x = ai->fln - 1;
  }
  /* Format: path/script <start:stop> <remote ip> <local ip> <remote netmask> */
  while(x < (int)ai->fln && x >= 0)
  {
    if(strstr(ai->filterid[x], ".."))
    {
      nsyslog(LOG_ERR, "Filter name %s is invalid.");
    }
    else
    {
      pid_t rc = fork();
      if(rc == -1)
      {
        nsyslog(LOG_ERR, "Can't fork for filter: %m");
        return 1;
      }
      if(rc) /* parent */
      {
        signal(SIGALRM, alrm_hand);
        alarm(2);
        wait(NULL);
        alarm(0);
      }
      else  /* child */
      {
        const char *args[10];
        char *buf = xmalloc(strlen(lineconf.filterdir) + strlen(ai->filterid[x]) + 2);

        int fd = open("/dev/null", O_RDWR);
        if(fd == -1)
        {
          nsyslog(LOG_ERR, "can't open /dev/null: %m");
          exit(1);
        }
        dup2(fd, 0);
        dup2(fd, 1);
        dup2(fd, 2);
        if(fd > 2)
          close(fd);

        sprintf(buf, "%s/%s", lineconf.filterdir, ai->filterid[x]);
        args[0] = buf;
        args[1] = islogin ? "start" : "stop";
        args[2] = xstrdup(dotted(ai->address));
        args[3] = xstrdup(dotted(ai->localip));
        args[4] = xstrdup(dotted(ai->netmask));
        args[5] = NULL;
        
        execv(args[0], (char **)args);
        nsyslog(LOG_ERR, "%s: %m", args[0]);
        exit(1);
      }
    }
    islogin ? x++ : x--;    /* FIFO add, LIFO del */
  }
  return 0;
}
