/*	Styled_Multiwriter

PIRL CVS ID: Styled_Multiwriter.java,v 1.12 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/
package	PIRL.Utilities;

import	java.io.Writer;
import	java.io.IOException;
import	java.util.Vector;
import	javax.swing.text.AttributeSet;

/**	A <i>Styled_Multiwriter</i> is a one-to-many Writer, any of which may
	be a Styled_Writer.
<p>
	Writers are wrapped in a Suspendable_Styled_Writer before being
	added to the list of Writers unless they are already a
	Suspendable_Styled_Writer.
<p>
	Each write method is repeated on each Writer in the Writers list.
	Each Writer is used even if a writer throws an exception. Any
	exceptions thrown are accumulated in a Multiwriter_IOException which
	associates the exception thrown with the original Writer source of
	the exception. A Writer that throws an exception is suspended, except
	in the case of the close method.
<p>
	<b>N.B.</b>: If the Writers list is empty each write method will do
	nothing.
<p>
	@author		Bradford Castalia - UA/PIRL
	@version	1.12
	@see		Writer
	@see		Styled_Writer
	@see		Suspendable_Styled_Writer
*/
public class Styled_Multiwriter
	extends Writer
	implements Styled_Writer
{
/**	Class identification name with source code version and date.
*/
public static final String
	ID = "PIRL.Utilities.Styled_Multiwriter (1.12 2012/04/16 06:18:24)";

/**	Determines whether a Writer is enabled or disabled when it is
	{@link #Add(Writer) add}ed to the Writers list.
*/
public volatile boolean
	Add_Enabled				= true;


//	List of Suspendable_Styled_Writer objects.
private Vector
	Writers					= new Vector ();
private Vector
	Unwrapped				= new Vector ();


private static final String
	NL			= System.getProperty ("line.separator");


//	DEBUG control.
private static final int
	DEBUG_OFF			= 0,
	DEBUG_CONSTRUCTOR	= 1 << 0,
	DEBUG_ACCESSORS		= 1 << 1,
	DEBUG_WRITE			= 1 << 2,
	DEBUG_ALL			= -1,

	DEBUG				= DEBUG_OFF;

/*==============================================================================
	Constructors
*/
/**	Construct a Styled_Multiwriter.
<p>
	Writers have yet to be {@link #Add(Writer) add}ed.
*/
public Styled_Multiwriter ()
{
if ((DEBUG & DEBUG_CONSTRUCTOR) != 0)
	System.out.println
		(">-< Styled_Multiwriter");
}

/*==============================================================================
	Accessors
*/
/**	Add a Writer to the list of {@link #Writers() Writers}.
<p>
	If the Writer is already in the list it is not added. A Writer that
	is added to the list is wrapped in a Suspendable_Styled_Writer unless
	it is already an instance of a Suspendable_Styled_Writer.
<p>
	The addition of the Writer to the Writers list is synchronized on
	the Writers list.
<p>
	@param	writer	The Writer to add.
	@return	true if the Writer was added to the list; false if it was
		already in the list or is null.
	@see	Suspendable_Styled_Writer
	@see	#Remove(Writer)
*/
public boolean Add
	(
	Writer	writer
	)
{
if (writer == null ||
	Contains (writer))
	return false;
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">>> Styled_Multiwriter.Add: " + writer);

if (writer instanceof Suspendable_Styled_Writer)
	Unwrapped.add (writer);
else
	writer = new Suspendable_Styled_Writer (writer, Add_Enabled);

synchronized (Writers)
	{Writers.add (writer);}
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		("<<< Styled_Multiwriter.Add: true");
return true;
}

/**	Remove a Writer from the list of {@link #Writers() Writers}.
<p>
	The removal of the Writer from the Writers list is synchronized on
	the Writers list.
<p>
	@param	writer	The Writer to remove. This is expected to be a
		Writer that was {@link #Add(Writer) Add}ed.
	@return	true if the Writer was removed to the list; false if it
		was not in the list or is null.
	@see	#Add(Writer)
	@see	#Entry_for(Writer)
*/
public boolean Remove
	(
	Writer	writer
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Styled_Multiwriter.Remove: " + writer);
if ((writer = Entry_for (writer)) == null)
	return false;

synchronized (Writers)
	{
	Writers.remove (writer);
	Unwrapped.remove (writer);
	}
return true;
}

/**	Remove all {@link #Writers() Writers} list entries.
<p>
	The removal of the Writer from the Writers list is synchronized on
	the Writers list.
<p>
	<b>N.B.</b>: The Writers entries are not {@link #close() closed}.
<p>
	@see	#Add(Writer)
*/
public void Remove_All ()
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Styled_Multiwriter.Remove_All");
synchronized (Writers)
	{Writers.clear ();}
}

/**	Turn output suspension on or off for a Writer.
<p>
	Searching the Writers list for the Writer is synchronized on the
	Writers list,
<p>
	@param	writer	The Writer to be affected. This is expected to be a
		Writer that was {@link #Add(Writer) Add}ed.
	@param	suspend	true if the Writer output is to be suspended; false
		if output is to be allowed.
	@return	true	If the Writer was found in the Writers list; false
		otherwise.
	@see	#Suspended(Writer)
	@see	#Entry_for(Writer)
*/
public boolean Suspend
	(
	Writer	writer,
	boolean	suspend
	)
{
if ((DEBUG & DEBUG_ACCESSORS) != 0)
	System.out.println
		(">-< Styled_Multiwriter.Suspend: " + writer + ' ' + suspend);
Suspendable_Styled_Writer
	suspendable_writer = Entry_for (writer);
if (suspendable_writer != null)
	{
	suspendable_writer.Suspend (suspend);
	return true;
	}
return false;
}

/**	Test if output for a Writer is suspended.
<p>
	Searching the Writers list for the Writer is synchronized on the
	Writers list,
<p>
	@param	writer	The Writer to test. This is expected to be a
		Writer that was {@link #Add(Writer) Add}ed.
	@return	true if the Writer output is suspended or could not be be
		found in the Writers list; false if the Writer is not suspended.
		<b>N.B.</b>: Because a Writer that is not in the Writers list
		will be deemed suspended - it will not produce any output via
		this object - test if the Writer is {@link #Contains(Writer)
		contained} in the list if the distinction is important.
	@see	#Suspend(Writer, boolean)
*/
public boolean Suspended
	(
	Writer	writer
	)
{
Suspendable_Styled_Writer
	suspendable_writer = Entry_for (writer);
if (suspendable_writer != null)
	return suspendable_writer.Suspended ();
return true;
}

/**	Test if the list of Writers contains a specified Writer.
<p>
	Searching the Writers list for the Writer is synchronized on the
	Writers list,
<p>
	@param	writer	The Writer to test for. This is expected to be a
		Writer that was {@link #Add(Writer) Add}ed.
	@return	true	If the Writer is in the Writers list; false otherwise.
	@see	#Entry_for(Writer)
*/
public boolean Contains
	(
	Writer	writer
	)
{return Entry_for (writer) != null;}

/**	Get the {@link #Writers() Writers} entry for a Writer that was
	{@link #Add(Writer) Add}ed.
<p>
	When a Writer is {@link #Add(Writer) Add}ed it is wrapped in a
	Suspendable_Styled_Writer if it is not already an instance of this
	class. This method searches for the Suspendable_Styled_Writer in the
	{@link #Writers() Writers} list that is associated with a specified
	Writer which may or may not have been wrapped by a
	Suspendable_Styled_Writer before being entered into the list.
<p>
	Searching the Writers list for the Writer is synchronized on the
	Writers list,
<p>
	@param	writer	A Writer. This is expected to be one of the Writer
		objects that was {@link #Add(Writer) Add}ed.
	@return	A Suspendable_Styled_Writer from the {@link #Writers() Writers}
		list. This will be null if the writer was not found.
*/
public Suspendable_Styled_Writer Entry_for
	(
	Writer	writer
	)
{
if (writer == null)
	return null;

synchronized (Writers)
	{
	int
		index = Writers.size ();
	while (--index >= 0)
		{
		Suspendable_Styled_Writer
			suspendable_writer = (Suspendable_Styled_Writer)Writers.get (index);
		if (writer == suspendable_writer ||
			writer == suspendable_writer.Writer ())
			return suspendable_writer;
		}
	}
return null;
}

/**	Get the Writer that was {@link #Add(Writer) Add}ed from a {@link
	#Writers() Writers} list entry.
<p>
	When a Writer is {@link #Add(Writer) Add}ed it is wrapped in a
	Suspendable_Styled_Writer if it is not already an instance of this
	class. This method searches for the Suspendable_Styled_Writer in the
	{@link #Writers() Writers} list that is associated with a specified
	Writer which may or may not have been wrapped by a
	Suspendable_Styled_Writer before being entered into the list. It then
	checks this against the list of Suspendable_Styled_Writer objects that
	were added without being wrapped. If this check succeeds then the
	Writer that was specified is returned; otherwise the
	{@link Suspendable_Styled_Writer#Writer() Writer} from this
	specified Suspendable_Styled_Writer is returned.
<p>
	Searching the Writers list is synchronized on the Writers list,
<p>
	@param	writer	A Suspendable_Styled_Writer from which to determine
		the original Writer that was {@link #Add(Writer) Add}ed.
	@return	The Writer that was {@link #Add(Writer) Add}ed, or null
		if the writer is not in the {@link #Writers() Writers} list.
*/
public Writer Entry_from
	(
	Suspendable_Styled_Writer	writer
	)
{
if (writer == null)
	return null;
synchronized (Writers)
	{
	if (! Writers.contains (writer))
		return null;
	if (Unwrapped.contains (writer))
		return writer;
	return writer.Writer ();
	}
}

/**	Test if the Writers list is empty.
<p>
	@return	true if the Writers list is empty; false otherwise.
*/
public boolean Is_Empty ()
{return Writers.isEmpty ();}


/**	Get a reference to the list of Writers.
<p>
	The Writers in the list are {@link Suspendable_Styled_Writer}
	objects. When a Writer is {@link #Add(Writer) Add}ed it is wrapped in
	a Suspendable_Styled_Writer if it is not already an instance of this
	class. A Suspendable_Styled_Writer always {@link
	Suspendable_Styled_Writer#Writer() contains a Writer} to which I/O is
	deferred.
<p>
	@return	A Vector of Suspendable_Styled_Writer objects. <b>N.B.</b>:
		The returned list is a reference, not a copy. <i>DO NOT MODIFY THE
		CONTENTS OF THIS LIST!</i>
	@see	#Add(Writer)
	@see	#Remove(Writer)
*/
public Vector Writers ()
{return Writers;}

/*==============================================================================
	Writer
*/
/**	Write a portion of an array of characters.
<p>
	Iterating over the Writers list is synchronized on the Writers list,
	and the write to the Writer is synchronized on the Writer.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	characters	The char array containing the characters to be
		written. If null or empty nothing is done.
	@param	offset	Array offset from which to start writing characters.
	@param	amount	The number of characters to write. If zero nothing is
		done.
	@throws	IndexOutOfBoundsException	If the offset or amount are
		negative or the offset plus the amount is greater than the length
		of the array.
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
	@see	#Suspended(Writer)
*/
public void write
	(
	char[]	characters,
	int		offset,
	int		amount
	)
	throws Multiwriter_IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">>> Styled_Multiwriter.write: " + offset + '/' + amount + " -" + NL
		+ characters);
if (characters == null ||
	characters.length == 0 ||
	amount == 0)
	return;

if (offset < 0 ||
	amount < 0 ||
	(offset + amount) > characters.length)
	throw new IndexOutOfBoundsException (ID + NL
		+ "Invalid values for " + characters.length
			+ " character array content -" + NL
		+ "  Array offset " + offset + " and amount " + amount + '.');

Multiwriter_IOException
	multiexception = null;
synchronized (Writers)
	{
	//	Write in the order of Writers entries.
	int
		index = -1,
		size = Writers.size ();
	while (++index < size)
		{
		Suspendable_Styled_Writer
			writer = (Suspendable_Styled_Writer)Writers.get (index);
		if ((DEBUG & DEBUG_WRITE) != 0)
			System.out.println
				("    Styled_Multiwriter: Writer " + writer.Writer ());
		try {synchronized (writer.Writer ())
			{writer.write (characters, offset, amount);}}
		catch (IOException exception)
			{
			if (multiexception == null)
				multiexception = new Multiwriter_IOException (ID + NL
					+ "Problem encountered while writing characters -" + NL
					+ new String (characters));
			multiexception.Sources.add (Entry_from (writer));
			multiexception.Exceptions.add (exception);
			writer.Suspend (true);
			}
		}
	}
if (multiexception != null)
	throw multiexception;
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		("<<< Styled_Multiwriter.write");
}

/**	Write an array of characters.
<p>
	Using this method is the same as using {@link #write(char[], int, int)
	<code>write (characters, 0, characters.length)</code>}.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	characters	The char array containing the characters to be
		written. If null or empty nothing is done.
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
*/
public void write
	(
	char[]	characters
	)
	throws Multiwriter_IOException
{
if (characters == null)
	return;
write (characters, 0, characters.length);
}

/**	Write a character.
<p>
	Iterating over the Writers list is synchronized on the Writers list,
	and the write to the Writer is synchronized on the Writer.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	character	The character to be written in the 16 low-order
		bits..
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
*/
public void write
	(
	int		character
	)
	throws Multiwriter_IOException
{
Multiwriter_IOException
	multiexception = null;
synchronized (Writers)
	{
	//	Write in the order of Writers entries.
	int
		index = -1,
		size = Writers.size ();
	while (++index < size)
		{
		Suspendable_Styled_Writer
			writer = (Suspendable_Styled_Writer)Writers.get (index);
		try {synchronized (writer.Writer ()) {writer.write (character);}}
		catch (IOException exception)
			{
			if (multiexception == null)
				multiexception = new Multiwriter_IOException (ID + NL
					+ "Problem encountered while writing character '"
						+ (char)character + "'.");
			multiexception.Sources.add (Entry_from (writer));
			multiexception.Exceptions.add (exception);
			writer.Suspend (true);
			}
		}
	}
if (multiexception != null)
	throw multiexception;
}

/**	Write a portion of a String.
<p>
	Using this method is the same as using {@link #write(char[], int, int)
	<code>write (string.toCharArray (), offset, amount)</code>}.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@param	offset	String offset from which to start writing characters.
	@param	amount	The number of characters to write. If zero nothing
		is done.
	@throws	IndexOutOfBoundsException	If the offset or amount are
		negative or the offset plus the amount is greater than the length
		of the string.
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
	@see	#write(char[], int, int)
*/
public void write
	(
	String			string,
	int				offset,
	int				amount
	)
	throws Multiwriter_IOException
{
if (string == null ||
	string.length () == 0 ||
	amount == 0)
	return;

if (offset < 0 ||
	amount < 0 ||
	(offset + amount) > string.length ())
	throw new IndexOutOfBoundsException (ID + NL
		+ "Invalid values for " + string.length ()
			+ " character String content -" + NL
		+ "  String offset " + offset + " and amount " + amount + '.');

write (string.toCharArray (), offset, amount);
}

/**	Write a String.
<p>
	Using this method is the same as using {@link #write(String, int, int)
	<code>write (string, 0, string.length ())</code>}.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	string	The String containing the characters to be written.
		If null or empty nothing is done.
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
	@see	#write(String, int, int)
*/
public void write
	(
	String			string
	)
	throws Multiwriter_IOException
{
if (string == null)
	return;
write (string, 0, string.length ());
}

/**	Flush the writers.
<p>
	Iterating over the Writers list is synchronized on the Writers list,
	and the flush to the Writer is synchronized on the Writer.
<p>
	All Writers are flushed even if one or more Writers throws an
	exception.
<p>
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
*/
public void flush ()
	throws Multiwriter_IOException
{
Multiwriter_IOException
	multiexception = null;
synchronized (Writers)
	{
	//	Write in the order of Writers entries.
	int
		index = -1,
		size = Writers.size ();
	while (++index < size)
		{
		Suspendable_Styled_Writer
			writer = (Suspendable_Styled_Writer)Writers.get (index);
		try {synchronized (writer.Writer ()) {writer.flush ();}}
		catch (IOException exception)
			{
			if (multiexception == null)
				multiexception = new Multiwriter_IOException (ID + NL
					+ "Problem encountered while flushing.");
			multiexception.Sources.add (Entry_from (writer));
			multiexception.Exceptions.add (exception);
			writer.Suspend (true);
			}
		}
	}
if (multiexception != null)
	throw multiexception;
}

/**	Close the writers.
<p>
	Iterating over the Writers list is synchronized on the Writers list,
	and the close of the Writer is synchronized on the Writer.
<p>
	@throws	Multiwriter_IOException	If any Writer throws an exception.
		All Writers are closed even if one or more Writers throws an
		exception. The Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed. <b>N.B.</b>: A Writer that
		throws an exception is not suspended.
*/
public void close ()
	throws Multiwriter_IOException
{
Multiwriter_IOException
	multiexception = null;
synchronized (Writers)
	{
	int
		index = Writers.size ();
	while (--index >= 0)
		{
		Suspendable_Styled_Writer
			writer = (Suspendable_Styled_Writer)Writers.get (index);
		try {synchronized (writer.Writer ()) {writer.close ();}}
		catch (IOException exception)
			{
			if (multiexception == null)
				multiexception = new Multiwriter_IOException (ID + NL
					+ "Problem encountered while closing.");
			multiexception.Sources.add (Entry_from (writer));
			multiexception.Exceptions.add (exception);
			}
		}
	}
if (multiexception != null)
	throw multiexception;
}

/*==============================================================================
	Styled_Writer
*/
/**	Write styled text.
<p>
	Iterating over the Writers list is synchronized on the Writers list,
	and the Write to the Writer is synchronized on the Writer.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	text	The text String to be written. If null or empty
		nothing is done.
	@param	style	The AttributeSet to be applied to the text. This
		may be null if plain text is to be displayed.
	@return	This Styled_Multiwriter
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
	@see	Suspendable_Styled_Writer#Write(String, AttributeSet)
	@see	Styled_Writer#Write(String, AttributeSet)
*/
public Styled_Writer Write
	(
	String			text,
	AttributeSet	style
	)
	throws Multiwriter_IOException
{
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		(">>> Styled_Multiwriter.Write:" + NL
		+"    Style: " + style + NL
		+ text);
		
if (text == null ||
	text.length () == 0)
	return this;

Multiwriter_IOException
	multiexception = null;
synchronized (Writers)
	{
	//	Write in the order of Writers entries.
	int
		index = -1,
		size = Writers.size ();
	while (++index < size)
		{
		Suspendable_Styled_Writer
			writer = (Suspendable_Styled_Writer)Writers.get (index);
		if ((DEBUG & DEBUG_WRITE) != 0)
			System.out.println
				("    Styled_Multiwriter: Writer " + writer.Writer ());
		try {synchronized (writer.Writer ()) {writer.Write (text, style);}}
		catch (IOException exception)
			{
			if (multiexception == null)
				multiexception = new Multiwriter_IOException (ID + NL
					+ "Problem encountered while Writing "
						+ ((style == null) ? "" : "styled ")
						+ "text -" + NL
					+ text);
			multiexception.Sources.add (Entry_from (writer));
			multiexception.Exceptions.add (exception);
			writer.Suspend (true);
			}
		}
	}
if (multiexception != null)
	throw multiexception;
if ((DEBUG & DEBUG_WRITE) != 0)
	System.out.println
		("<<< Styled_Multiwriter.Write");
return this;
}

/**	Write plain text.
<p>
	This is the same as {@link #Write(String, AttributeSet) writing}
	text with a null style.
<p>
	All Writers are written even if one or more Writers throws an
	exception.
<p>
	@param	text	The text String to be written.
	@return	This Styled_Multiwriter
	@throws	Multiwriter_IOException	If any Writer throws an exception. A
		Writer that throws an exception is {@link
		Suspendable_Styled_Writer#Suspend(boolean) suspend}ed. The
		Writers in the exception's {@link
		Multiwriter_IOException#Sources} list are the original Writers
		that were {@link #Add(Writer) Add}ed.
*/
public Styled_Writer Write
	(
	String			text
	)
	throws Multiwriter_IOException
{return Write (text, null);}

}
