/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment interactive test using PgmImage.
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <tests/check/common/pgmcheck.h>
#include <pgm/pgm.h>
#include "interactive_tests.h"


static void
on_image_loaded (PgmImage *image, gpointer  user_data)
{
  pgm_interactive_test_rendering_done ();
}

static PgmDrawable *
create_image (guint grid_width, guint grid_height, guint x, guint y)
{
  PgmDrawable *img = NULL;
  gfloat width = 0.0f, height = 0.0f, space = 0.0f;
  width = 3.75f / (float) grid_width;
  height = 2.75f / (float) grid_height;
  space = 0.25f / (float) (grid_width + 1);

  pgm_interactive_test_asynchronous_rendering ();
  img = pgm_image_new ();
  g_signal_connect (G_OBJECT (img), "pixbuf_loaded",
                    G_CALLBACK (on_image_loaded), NULL);
  pgm_image_set_from_file (PGM_IMAGE (img), PGM_TEST_IMAGE_FILE, 0);

  pgm_drawable_set_size (img, width, height);
  pgm_drawable_set_position (img,
                             space + (width + space) * x,
                             space + (height + space) * y,
                             0.0f);
  pgm_drawable_set_bg_color (img, 20, 20, 20, 255);
  pgm_drawable_show (img);

  return img;
}

PGM_START_INTERACTIVE_TEST (test_image_basic)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img = NULL;

  canvas = pgm_canvas_new ();

  img = create_image (1, 1, 0, 0);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
  pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img);

}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_image_alignment_zoomed)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img[9] = { NULL };
  guint i = 0;

  canvas = pgm_canvas_new ();

  for (i = 0; i< 9; i++)
    {
      img[i] = create_image (3, 3, i%3, i/3);
      pgm_image_set_layout (PGM_IMAGE (img[i]), PGM_IMAGE_ZOOMED);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img[i]);
    }

  pgm_image_set_alignment (PGM_IMAGE (img[0]), PGM_IMAGE_TOP_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[1]), PGM_IMAGE_TOP_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[2]), PGM_IMAGE_TOP_RIGHT);
  pgm_image_set_alignment (PGM_IMAGE (img[3]), PGM_IMAGE_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[4]), PGM_IMAGE_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[5]), PGM_IMAGE_RIGHT);
  pgm_image_set_alignment (PGM_IMAGE (img[6]), PGM_IMAGE_BOTTOM_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[7]), PGM_IMAGE_BOTTOM_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[8]), PGM_IMAGE_BOTTOM_RIGHT);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_image_alignment_scaled_horiz)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img[9] = { NULL };
  guint i = 0;

  canvas = pgm_canvas_new ();

  for (i = 0; i< 9; i++)
    {
      img[i] = create_image (3, 3, i%3, i/3);
      pgm_image_set_layout (PGM_IMAGE (img[i]), PGM_IMAGE_SCALED);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img[i]);
    }

  pgm_image_set_alignment (PGM_IMAGE (img[0]), PGM_IMAGE_TOP_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[1]), PGM_IMAGE_TOP_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[2]), PGM_IMAGE_TOP_RIGHT);
  pgm_image_set_alignment (PGM_IMAGE (img[3]), PGM_IMAGE_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[4]), PGM_IMAGE_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[5]), PGM_IMAGE_RIGHT);
  pgm_image_set_alignment (PGM_IMAGE (img[6]), PGM_IMAGE_BOTTOM_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[7]), PGM_IMAGE_BOTTOM_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[8]), PGM_IMAGE_BOTTOM_RIGHT);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_image_alignment_scaled_vert)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img[9] = { NULL };
  guint i = 0;

  canvas = pgm_canvas_new ();

  for (i = 0; i< 9; i++)
    {
      img[i] = create_image (5, 3, (i%3) + 1, i/3);
      pgm_image_set_layout (PGM_IMAGE (img[i]), PGM_IMAGE_SCALED);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img[i]);
    }

  pgm_image_set_alignment (PGM_IMAGE (img[0]), PGM_IMAGE_TOP_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[1]), PGM_IMAGE_TOP_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[2]), PGM_IMAGE_TOP_RIGHT);
  pgm_image_set_alignment (PGM_IMAGE (img[3]), PGM_IMAGE_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[4]), PGM_IMAGE_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[5]), PGM_IMAGE_RIGHT);
  pgm_image_set_alignment (PGM_IMAGE (img[6]), PGM_IMAGE_BOTTOM_LEFT);
  pgm_image_set_alignment (PGM_IMAGE (img[7]), PGM_IMAGE_BOTTOM_CENTER);
  pgm_image_set_alignment (PGM_IMAGE (img[8]), PGM_IMAGE_BOTTOM_RIGHT);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_image_layout)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *img[3] = { NULL };
  guint i = 0;

  canvas = pgm_canvas_new ();

  for (i = 0; i< 3; i++)
    {
      img[i] = create_image (3, 3, i%3, 1);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, img[i]);
    }

  pgm_image_set_layout (PGM_IMAGE (img[0]), PGM_IMAGE_FILLED);
  pgm_image_set_layout (PGM_IMAGE (img[1]), PGM_IMAGE_SCALED);
  pgm_image_set_layout (PGM_IMAGE (img[2]), PGM_IMAGE_ZOOMED);

  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

PGM_START_INTERACTIVE_TEST (test_image_alpha_render)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *image[4] = { NULL };
  guint i = 0;

  canvas = pgm_canvas_new ();

  for (i=0; i<4; i++)
    {
      image[i] = create_image (2, 2, i%2, i/2);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, image[i]);
      pgm_drawable_set_opacity (image[i], i?(64 * i - 1):0);
    }


  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

/* This checks that the stuff introduced in r1069 is working */
PGM_START_INTERACTIVE_TEST (test_image_alpha_render_noblend)
{
  PgmCanvas *canvas = NULL;
  PgmDrawable *image[4] = { NULL };
  guint i = 0;

  pgm_viewport_set_alpha_blending (interactive_test_viewport, FALSE);
  canvas = pgm_canvas_new ();

  for (i=0; i<4; i++)
    {
      image[i] = create_image (2, 2, i%2, i/2);
      pgm_canvas_add (canvas, PGM_DRAWABLE_MIDDLE, image[i]);
      pgm_drawable_set_opacity (image[i], i?(64 * i - 1):0);
    }


  pgm_viewport_set_canvas (interactive_test_viewport, canvas);
}
PGM_END_INTERACTIVE_TEST

Suite *
pgm_image_suite (void)
{
  Suite *s = suite_create ("PgmImage");
  TCase *tc_chain = tcase_create ("pgmimage tests");

  PGM_INTERACTIVE_TESTS_SET_FIXTURES (tc_chain);

  tcase_set_timeout (tc_chain, 500);

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_image_basic);
  tcase_add_test (tc_chain, test_image_alignment_zoomed);
  tcase_add_test (tc_chain, test_image_alignment_scaled_horiz);
  tcase_add_test (tc_chain, test_image_alignment_scaled_vert);
  tcase_add_test (tc_chain, test_image_layout);
  tcase_add_test (tc_chain, test_image_alpha_render);
  tcase_add_test (tc_chain, test_image_alpha_render_noblend);

  return s;
}

