/*
 * Copyright © INRIA 2009
 * Brice Goglin <Brice.Goglin@inria.fr>
 *
 * This software is a computer program whose purpose is to provide
 * a fast inter-process communication subsystem.
 *
 * This software is governed by the CeCILL-B license under French law and
 * abiding by the rules of distribution of free software.  You can  use,
 * modify and/ or redistribute the software under the terms of the CeCILL-B
 * license as circulated by CEA, CNRS and INRIA at the following URL
 * "http://www.cecill.info".
 *
 * As a counterpart to the access to the source code and  rights to copy,
 * modify and redistribute granted by the license, users are provided only
 * with a limited warranty  and the software's author,  the holder of the
 * economic rights,  and the successive licensors  have only  limited
 * liability.
 *
 * In this respect, the user's attention is drawn to the risks associated
 * with loading,  using,  modifying and/or developing or reproducing the
 * software by the user in light of its specific status of free software,
 * that may mean  that it is complicated to manipulate,  and  that  also
 * therefore means  that it is reserved for developers  and  experienced
 * professionals having in-depth computer knowledge. Users are therefore
 * encouraged to load and test the software's suitability as regards their
 * requirements in conditions enabling the security of their systems and/or
 * data to be ensured and,  more generally, to use and operate it in the
 * same conditions as regards security.
 *
 * The fact that you are presently reading this means that you have had
 * knowledge of the CeCILL-B license and that you accept its terms.
 */

#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <sched.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include "knem_io.h"

#define LEN 5000000

#define SENDER_IGNORED_CHAR 'a'
#define SENDER_CHAR 'b'
#define RECEIVER_IGNORED_CHAR 'c'
#define RECEIVER_CHAR 'c'

int main(int argc, char * argv[])
{
  struct knem_cmd_info info;
  int fd, err;
  char *send_buffer, *recv_buffer;
  struct knem_cmd_param_iovec send_iovec[5], recv_iovec[5];
  struct knem_cmd_init_send_param sendcmd;
  struct knem_cmd_sync_recv_param recvcmd;
  int i,j, k;
  int last_copied;

  send_buffer = malloc(LEN);
  if (!send_buffer)
    goto out;
  recv_buffer = malloc(LEN);
  if (!recv_buffer)
    goto out_with_send_buffer;

  send_iovec[0].base = (uintptr_t) send_buffer + 53;
  send_iovec[0].len = 7; /* total = 7 */
  send_iovec[1].base = (uintptr_t) send_buffer + 100;
  send_iovec[1].len = 253; /* total = 260 */
  send_iovec[2].base = (uintptr_t) send_buffer + 1000;
  send_iovec[2].len = 14567; /* total = 14827 */
  send_iovec[3].base = (uintptr_t) send_buffer + 20000;
  send_iovec[3].len = 123456; /* total = 138283 */
  send_iovec[4].base = (uintptr_t) send_buffer + 1000000;
  send_iovec[4].len = 3333333; /* total = 3471616 */

  recv_iovec[0].base = (uintptr_t) recv_buffer + 10;
  recv_iovec[0].len = 3333333; /* total = 3333333 */
  recv_iovec[1].base = (uintptr_t) recv_buffer + 3500000;
  recv_iovec[1].len = 12345; /* total = 3345678 */
  recv_iovec[2].base = (uintptr_t) recv_buffer + 3600000;
  recv_iovec[2].len = 253; /* total = 3345931 */
  recv_iovec[3].base = (uintptr_t) recv_buffer + 3700000;
  recv_iovec[3].len = 86523; /* total = 3432454 */
  recv_iovec[4].base = (uintptr_t) recv_buffer + 3800000;
  recv_iovec[4].len = 7; /* total = 3432461 */

  /* the last 39155 bytes of recv_iovec won't be touched */

  fd = open("/dev/knem", O_RDWR);
  if (fd < 0) {
    perror("open");
    goto out_with_recv_buffer;
  }

  err = ioctl(fd, KNEM_CMD_GET_INFO, &info);
  if (err < 0) {
    perror("ioctl (get info)");
    goto out_with_fd;
  }

  if (info.abi != KNEM_ABI_VERSION) {
    printf("got driver ABI %lx instead of %lx\n",
	   (unsigned long) info.abi, (unsigned long) KNEM_ABI_VERSION);
    goto out_with_fd;
  }

  printf("got driver ABI %lx and feature mask %lx\n",
	 (unsigned long) info.abi, (unsigned long) info.features);

  memset(send_buffer, SENDER_IGNORED_CHAR, LEN);
  for(i=0; i<5; i++)
    memset((void *)(uintptr_t) send_iovec[i].base, SENDER_CHAR, send_iovec[i].len);
  memset(recv_buffer, RECEIVER_IGNORED_CHAR, LEN);
  for(i=0; i<5; i++)
    memset((void *)(uintptr_t) recv_iovec[i].base, RECEIVER_CHAR, recv_iovec[i].len);

  sendcmd.send_iovec_array = (uintptr_t) send_iovec;
  sendcmd.send_iovec_nr = 5;
  sendcmd.flags = 0;
  err = ioctl(fd, KNEM_CMD_INIT_SEND, &sendcmd);
  if (err < 0) {
    perror("ioctl (init send)");
    goto out_with_fd;
  }
  printf("got lid %llx\n", (unsigned long long) sendcmd.send_cookie);

  recvcmd.recv_iovec_array = (uintptr_t) recv_iovec;
  recvcmd.recv_iovec_nr = 5;
  recvcmd.send_cookie = sendcmd.send_cookie;
  recvcmd.flags = 0;
  err = ioctl(fd, KNEM_CMD_SYNC_RECV, &recvcmd);
  if (err < 0) {
    perror("ioctl (init sync recv)");
    goto out_with_fd;
  }

  if (recvcmd.status != KNEM_STATUS_SUCCESS) {
    fprintf(stderr, "got status %d instead of %d\n", recvcmd.status, KNEM_STATUS_SUCCESS);
    goto out_with_fd;
  }

  /* check that the receiver contains either unchanged or the sender char */

  for(i=0; i<LEN; i++)
    if (recv_buffer[i] != RECEIVER_CHAR && recv_buffer[i] != RECEIVER_IGNORED_CHAR && recv_buffer[i] != SENDER_CHAR)
      printf("invalid char at offset %d, got %c\n", i, recv_buffer[i]);

  last_copied = -1;
  k=0;
  for(j=0; j<5; j++) {
    char * buffer = (void *)(uintptr_t) recv_iovec[j].base;
    for(i=0; i<recv_iovec[j].len; i++, k++) {
      if (buffer[i] != RECEIVER_CHAR && buffer[i] != SENDER_CHAR)
	printf("invalid char at offset %d segment %d, got %c\n", i, j, buffer[i]);
      if (buffer[i] == SENDER_CHAR) {
	if (last_copied != k-1)
	  printf("got a modified char at %d while the last one was at %d\n", k, last_copied-1);
	last_copied = k;
      }
    }
  }
  printf("last copied at %d\n", last_copied);

 out_with_fd:
  close(fd);
 out_with_recv_buffer:
  free(recv_buffer);
 out_with_send_buffer:
  free(send_buffer);
 out:
  return 0;
}
