/* cdw
 * Copyright (C) 2002 Varkonyi Balazs
 * Copyright (C) 2007 - 2012 Kamil Ignacak
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _GNU_SOURCE /* strndup() */

/**
   \file cdw_window.c
*/


#include <string.h>
#include <stdlib.h>

#include "cdw_window.h"
#include "cdw_string.h"
#include "cdw_debug.h"





/**
   \brief Create new or derived ncurses window

   Function creates new (newwin()) or derived (derwin()) ncurses window
   and draws a box around it (possibly with top and bottom titles).

   \p parent NULL pointer if you want to create new window with newwin();
   pointer to parent window if you want to create derived window with derwin()

   \p n_lines
   \p n_cols

   \p begin_y - beginning of window, relative to stdscr for new window,
   \p begin_x  - beginning of window, relative to origin of parent window for derived window

   \p cdw_colors

   window is boxed with cdw_ncurses_nice_box(), these two strings are arguments to the function
   \p top_string
   \p bottom_string

   \return NULL on errors
   \return pointer to new window on success
*/
WINDOW *cdw_window_new(WINDOW *parent, int n_lines, int n_cols, int begin_y, int begin_x, chtype cdw_colors, const char *top_string, const char *bottom_string)
{
	WINDOW *window = (WINDOW *) NULL;
	if (parent == (WINDOW *) NULL) {
		window = newwin(n_lines, n_cols, begin_y, begin_x);
	} else {
		window = derwin(parent, n_lines, n_cols, begin_y, begin_x);
	}
	if (window == (WINDOW *) NULL) {
		cdw_vdm ("ERROR: failed to create %s window\n",
			 parent == (WINDOW *) NULL ? "new" : "derived");
		cdw_vdm ("ERROR: problem occurred when creating \"%s\" / \"%s\" window\n",
			 top_string, bottom_string);
		return (WINDOW *) NULL;
	}

	wbkgd(window, COLOR_PAIR(cdw_colors));
	int rv = keypad(window, TRUE);
	if (rv != OK) {
		cdw_vdm ("ERROR: keypad() returns !OK, but continuing\n");
		cdw_vdm ("ERROR: problem occurred when creating \"%s\" / \"%s\" window\n",
			 top_string, bottom_string);
	}
	rv = werase(window);
	if (rv != OK) {
		cdw_vdm ("ERROR: werase() returns !OK, but continuing\n");
		cdw_vdm ("ERROR: problem occurred when creating \"%s\" / \"%s\" window\n",
			 top_string, bottom_string);
	}

	if (parent != (WINDOW *) NULL) {
		return window;
	}
	cdw_rv_t crv = cdw_window_add_strings(window, top_string, bottom_string);
	if (crv != CDW_OK) {
		cdw_vdm ("ERROR: cdw_ncurses_nice_box() returns !CDW_OK\n");
		cdw_vdm ("ERROR: problem occurred when creating \"%s\" / \"%s\" window\n",
			 top_string, bottom_string);
		delwin(window);
		window = (WINDOW *) NULL;
	} else {
		rv = wrefresh(window);
		if (rv != OK) {
			cdw_vdm ("ERROR: wrefresh() returns !OK, but continuing\n");
			cdw_vdm ("ERROR: problem occurred when creating \"%s\" / \"%s\" window\n",
				 top_string, bottom_string);
		}
	}

	return window;
}





/**
   \brief Delete ncurses window, set pointer to NULL

   This is such a common piece of code that I had to put it in
   a separate function.
   The function accepts pointer to NULL window.
   The function sets window pointer to NULL.

   \param window - window to delete
*/
void cdw_window_delete(WINDOW **window)
{
	if (*window == (WINDOW *) NULL) {
		cdw_vdm ("WARNING: passing NULL window to the function\n");
		return;
	} else {
		delwin(*window);
		*window = (WINDOW *) NULL;
		return;
	}
}





/**
   \brief Add '?' char in upper right corner of given window

   The char indicates that pressing '?' key in this window invokes
   small help window, displaying some help text.

   \param window - window for which to display the char
   \param n_cols - number of columns in the window
*/
void cdw_window_add_help(WINDOW *window)
{
	int n_cols = getmaxx(window);
	mvwprintw(window, 0, n_cols - 1, "?");
	mvwaddch(window, 0, n_cols - 2, ACS_RTEE);
	mvwaddch(window, 1, n_cols - 1, ACS_TTEE);

	return;
}





/**
   \brief Draw nice box around window, with label on top and at the bottom of window

   Both top_string and bottom_string are optional and can be set to NULL
   or "" string.

   If the strings are longer than window width, then they will be truncated.
   Three dots will be appended to end of top string and/or to front of bottom string.

   \param window - initialized ncurses window to be boxed
   \param top_string - string that will appear on top of window
   \param bottom_string - string that will appear on bottom of window

   \return CDW_ERROR on malloc errors
   \return CDW_OK when no problems occurred
*/
cdw_rv_t cdw_window_add_strings(WINDOW *window, const char *top_string, const char *bottom_string)
{
	cdw_assert (window, "cannot box NULL window\n");

	/* 6 cells: left corner, left paren, space, <here be string>, space,
	   right paren, right corner - that is 6 chars in top and bottom
	   border of window, which should be preserved and can't be
	   overwritten by top_string nor bottom string */
	cdw_assert (getmaxx(window) > 6, "window width is 6 or less, it is a tad to small\n");
	size_t usable_n_cols = (size_t) getmaxx(window) - 6;

	/* draw standard ncurses box, and then overwrite parts of
	   border with string(s) and custom decoration */

	box(window, 0, 0);

	/* If bottom string will be truncated, then truncation should occur
	   at the beginning of the string. This is because when nice_box()
	   is used to box a file system browser window, bottom string
	   in file browser window represents current full directory path,
	   and truncation of beginning of path (while keeping it's end with
	   current directory intact) is better than the second possibility. */
	if ( bottom_string && strcmp(bottom_string, "") ) {
		bool malloced = false;
		char *buf = (char *) NULL;
		size_t len = strlen(bottom_string);

		if (len > usable_n_cols) {
			/* this is how far you have to move start of bottom
			   string to display correctly tail of bottom string */
			size_t offset = len - usable_n_cols;
			buf = strdup(bottom_string + offset);
			if (!buf) {
				cdw_vdm ("ERROR: failed to duplicate bottom string \"%s\" with strdup(bottom_string + offset)\n", bottom_string);
				cdw_vdm ("ERROR: details: strlen(bottom_string) = %zd, offset = %zd\n", strlen(bottom_string), offset);
				return CDW_ERROR;
			} else {
				malloced = true;
				strncpy(buf, "...", 3);
			}
		} else {
			malloced = false;
			/* display original argument string */
			buf = bottom_string;
		}
		mvwprintw(window, getmaxy(window) - 1, 1, "( %s )", buf);
		if (malloced) {
			free(buf);
			buf = (char *) NULL;
		}
	}

	if ( top_string && strcmp(top_string, "") ) {
		bool malloced = false;
		char *buf = (char *) NULL;
		size_t len = strlen(top_string);

		if (len > usable_n_cols) {
			/* we can display no more than usable_n_cols chars,
			   counting from start of string */
			buf = strndup(top_string, usable_n_cols);
			if (!buf) {
				cdw_vdm ("ERROR: failed to duplicate top string \"%s\" with strndup(top_string, usable_n_cols)\n", top_string);
				cdw_vdm ("ERROR: details: strlen(top_string) = %zd, usable_n_cols = %zd\n", strlen(top_string), usable_n_cols);
				return CDW_ERROR;
			} else {
				malloced = true;
				strncpy(buf + usable_n_cols - 3, "...", 3);
				*(buf + usable_n_cols) = '\0';
			}
		} else {
			malloced = false;
			/* display original string, without any modifications */
			buf = top_string;
		}

		mvwaddch(window, 0, 1, ACS_RTEE);
		mvwprintw(window, 0, 2, " %s ", buf);
		waddch(window, ACS_LTEE);
		if (malloced) {
			free(buf);
			buf = (char *) NULL;
		}
	}

	return CDW_OK;
}





/**
   \brief Print message in given window, wrap message to not to cross window border

   Print given \p message in a \p window. Text can be longer than window
   width. The function takes care of wrapping the text so that line breaks
   occur on white chars. Line breaks ('\n' chars) that exist in original
   message will be preserved.

   Caller of the function must make sure that window has correct height
   and width, e.g. by calling cdw_textarea_calculate_height().

   Caller must refresh \p window after calling this function.

   \param window - area in which you want to print the message
   \param message - message that has to be printed in a window
   \param align - alignment of message in window (CDW_ALIGN_CENTER, CDW_ALIGN_LEFT, CDW_ALIGN_RIGHT)

   \return -1 on errors
   \return 0 if \p window has just enough or more rows than needed to print a message without truncations
   \return number of lines that weren't displayed because \p window doesn't have enough rows
 */
int cdw_window_print_message(WINDOW *window, const char *message, int align)
{
	cdw_assert (window != (WINDOW *) NULL, "ERROR: cannot print in NULL window\n");
	cdw_assert (message != (char *) NULL, "ERROR: cannot print NULL message\n");

	int n_cols = getmaxx(window);
	int n_rows = getmaxy(window);
	int n_lines = (int) cdw_string_count_lines(message, (size_t) n_cols);
	if (n_lines > n_rows) {
		cdw_vdm ("WARNING: message has more lines (%d) than number of rows in the window (%d)\n",
			 n_lines, n_rows);
		cdw_vdm ("WARNING: \"%s\"\n", message);
	}

	char *string = cdw_string_wrap(message, (size_t) n_cols, align);
	if (string == (char *) NULL) {
		cdw_vdm ("ERROR: can't produce wrapped string\n");
		return -1;
	}
	mvwprintw(window, 0, 0, string);

	free(string);
	string = (char *) NULL;

	if (n_rows < n_lines) {
		cdw_vdm ("ERROR: function returns %d, there were more lines of text than rows in window\n",
			 n_lines - n_rows);
		return n_lines - n_rows;
	} else {
		return 0;
	}
}



