/**
 * @brief Initialization code for pbbuttonsd
 *
 * This file contains basic functions to initialize the daemon
 * pbbuttonsd. This starts with filling basic data structures,
 * evaluating command line arguments and does not end with signal
 * handlers.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/init.c
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include "systems.h"

#include <stdio.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <getopt.h>
#include <errno.h>
#include <pwd.h>
#include <glib.h>
#include <pbb.h>

#include "init.h"
#include "gettext_macros.h"
#include "input_manager.h"

#ifdef WITH_MODULE_PMAC
#  include "module_pmac.h"
#endif
#ifdef WITH_MODULE_ACPI
#  include "module_acpi.h"
#endif
#ifdef WITH_MODULE_IMAC
#  include "module_imac.h"
#endif

#include "class_backlight.h"
#include "class_mixer.h"
#include "class_config.h"

#include "module_system.h"
#include "module_powersave.h"
#include "module_display.h"
#include "module_cdrom.h"
#ifdef DEBUG
#  include "module_peep.h"
#endif

/**
 * @brief Initialize the main data structure
 *
 * This function is the main initialization function of pbbuttonsd.
 * It sets up tha main data structure. This data structure contains
 * a list of all modules. The sequence to initialize the modules is
 * defined with this list.
 *
 * @param  *sd  pointer to the main data structure
 */
void
init_serverdata (struct serverdata *sd)
{
	int n = 0;

	sd->prgname	   = NULL;
	sd->quiet      = 0;
	sd->configfile = DEFAULT_CONFIG;
	sd->mainloop   = NULL;

	sd->module[n].initfunc = inputmanager_init;  sd->module[n++].exitfunc    = inputmanager_exit;  /* pri 0 */
	sd->module[n].initfunc = ipc_init_stub;      sd->module[n++].exitfunc    = ipc_exit;           /* pri 0 */
	sd->module[n].initfunc = system_init;        sd->module[n++].exitfunc    = system_exit;        /* pri 1 */
	sd->module[n].initfunc = backlight_init;     sd->module[n++].exitfunc    = backlight_exit;     /* pri 1 */
#ifdef WITH_MODULE_PMAC
	sd->module[n].initfunc = pmac_init;          sd->module[n++].exitfunc    = pmac_exit;          /* pri 1 */
#endif
#ifdef WITH_MODULE_ACPI
	sd->module[n].initfunc = acpi_init;          sd->module[n++].exitfunc    = acpi_exit;          /* pri 1 */
#endif
#ifdef WITH_MODULE_IMAC
	sd->module[n].initfunc = imac_init;          sd->module[n++].exitfunc    = imac_exit;          /* pri 1 */
#endif
	sd->module[n].initfunc = power_init;         sd->module[n++].exitfunc    = power_exit;         /* pri 2 */
	sd->module[n].initfunc = display_init;       sd->module[n++].exitfunc    = display_exit;       /* pri 2 */
	sd->module[n].initfunc = cdrom_init;         sd->module[n++].exitfunc    = cdrom_exit;         /* pri 2 */
	sd->module[n].initfunc = mixer_init;         sd->module[n++].exitfunc    = mixer_exit;         /* pri 2 */
#ifdef DEBUG
	sd->module[n].initfunc = peep_init;          sd->module[n++].exitfunc    = peep_exit;          /* pri 3 */
#endif

	for (; n < MODULECOUNT; n++) {
		sd->module[n].initfunc = NULL;
		sd->module[n].exitfunc = NULL;
	}
}

/**
 * @brief initializes all pbbuttonsd modules
 *
 * This function calls every registered module to initialize
 * themselves. If the initialization of a module failed,  all
 * already loaded modules would be cleaned up.
 *
 * @param  sd  pointer to system data pool, that contains the 
 *             module list
 * @return  error code or zero if everything went well
 */
int
init_modules(struct serverdata *sd)
{
	int n, rc = 0;

	for (n=0; n < MODULECOUNT; n++)
		if (sd->module[n].initfunc != NULL)
			if ((rc = sd->module[n].initfunc ())) {
				for (--n; n >= 0; n--)   /* cleanup already initialized modules */
					if (sd->module[n].exitfunc != NULL)
						sd->module[n].exitfunc ();
				break; /* critical error - program abortion */
			}

	return rc;
}

/**
 * @brief  free all allocated ressources and cleanup the modules
 *
 * This funcion calls the exit funcion of every know modules. The sequence
 * is from last to first, that means the module that was initialized first
 * will be cleaned up last.
 *
 * @param  sd  pointer to system data pool, that contains the 
 *             module list
 */
void
exit_modules (struct serverdata *sd)
{
	int n;

	for (n = MODULECOUNT-1; n >= 0; n--)
		if (sd->module[n].exitfunc != NULL)
			sd->module[n].exitfunc ();
}

/** 
 * @brief evaluate command line arguments 
 *
 * This function read additional options from the command line, parse
 * them and fill the data into the system data pool.
 *
 * @param  sd      pointer to system data pool
 * @param  argc    count of given command line arguments
 * @param  argv[]  null-terminated array of command line arguments
 * @return error code or zero if everthing went well
 */
int
evaluate_args(struct serverdata *sd, int argc, char *argv[])
{
	struct option const long_options[] = {
		  {"help", no_argument, 0, ARG_HELP},
		  {"version", no_argument, 0, ARG_VERSION},
		  {"detach", optional_argument, 0, ARG_DETACH},
		  {"quiet", no_argument, 0, ARG_QUIET},
		  {"configfile", required_argument, 0, ARG_CONFIG},
		  {NULL, 0, NULL, 0}
	};
	int c, err;
	char *optarg2;

	if((sd->prgname = strrchr(argv[0],'/')) == NULL)
		sd->prgname = argv[0];
	else sd->prgname++;		/* ignore first slash*/
	argv[0] = sd->prgname;

	while ((c = getopt_long (argc, argv, ARG_ALL, long_options, (int *) 0)) != EOF) {
		switch (c) {
			case ARG_VERSION:
				printf(_("%s, version %s"), PACKAGE, VERSION);
				printf(", (c) 2002-2006 Matthias Grimm\n");
				return E_INFO;
			case ARG_DETACH:
				optarg2 = (optarg) ? optarg : argv[optind];
				if (optarg2 == NULL || optarg2[0] == '-')
					optarg2 = DEFAULT_PIDFILE;
				prepare_daemon (sd->prgname, optarg2, PBBDF_FORCE);
				break;
			case ARG_QUIET:
				sd->quiet = 1;
				break;
			case ARG_CONFIG:
				if ((err = check_devorfile(sd->configfile = optarg, TYPE_FILE))) {
					printf (_("ERROR: Have problems reading configuration file [%s]: %s\n"), sd->configfile, strerror(errno));
					return err;
				}
				break;
			case ARG_HELP:
			default:
				printf(_("%s - daemon to support special features of laptops and notebooks.\n"), sd->prgname);
				printf(_("Usage: %s [OPTION]... \n"), sd->prgname);
				printf (_("Options:\n"
					"   -%c, --help               display this help text and exit\n"
					"   -%c, --version            display version information and exit\n"
					"   -%c, --quiet              suppress welcome message\n"
					"   -%c, --detach[=PIDFILE]   start %s as background process and\n"
					"                            optional use an alternative pid-file\n"
					"                            (default: %s)\n"
					"   -%c, --configfile=CONFIG  use alternative configuration file\n"
					"                            (default: %s)\n"
					"see configuration file for more options.\n"),
					ARG_HELP, ARG_VERSION, ARG_QUIET, ARG_DETACH, sd->prgname, DEFAULT_PIDFILE,
					ARG_CONFIG, sd->configfile);
				return E_INFO;
		}
	}
	return 0;
}

/**
 * @brief  Static pipe to connect the two-part signal handler
 *
 * Pipe must be global because there is no other way for the
 * signal handler to access this data
 *
 * @static SigHandlerPipe[]
 */
static int SigHandlerPipe[2];

/**
 * @brief  Generic signal handler for all catched signals
 *
 * This handler is a generic signal handler which is set up to
 * all catched signals. The handler sends the signal numnber
 * through a pipe to a other signal handler which is running
 * in the program context. With this techique the signal handler
 * is very fast and the real signal handler can do things a
 * signal handler can't.
 *
 * @param  signum   signal number
 */
void
cbSigHandlerTX (int signum)
{
	write (SigHandlerPipe[1], &signum, sizeof(int));
}

/**
 * @brief  Second part of the signal handler
 *
 * This function is the other end of the pipe. This signal handler
 * received the signal numbers from the signal handler, interpretes
 * the signal and perform the appropriate action.
 *
 * @param  fd         file handle of the signal pipe
 * @param  user_data  pointer to main data structure
 * @return TRUE, if the handler should be kept active or
 *         FALSE when the handler should terminate.
 */
gboolean
cbSigHandlerRD (int fd, gpointer user_data)
{
	struct serverdata *base = user_data;
	int signum;
	gboolean rc = FALSE;

	if ((read (fd, &signum, sizeof(int))) == sizeof(int)) {
		switch (signum) {
			case SIGHUP:     /* reload config file and reconfigure the modules */
				scanEventDevices ();   /* look which HID are available */
				rc = TRUE;
				break;
			case SIGUSR1:    /* reset hardware configuration */
				process_queue_single (CONFIGQUEUE, TAG_REINIT, 0);
				rc = TRUE;
				break;
			case SIGINT:
			case SIGTERM:    /* cleanup and exit the program */
				close (SigHandlerPipe[0]);
				close (SigHandlerPipe[1]);
				g_main_loop_quit (base->mainloop);
				break;
		}
	}
	return rc;
}

/**
 * @brief  Install signal handlers for some signals
 *
 * This function creates the pipe to connect the two-part
 * signal handler and installs signal handlers for the
 * signals SIGINT, SIGTERM, SIGHUP and SIGUSR1.
 *
 * The initialized pipe is stored in the global variable
 * SigHandlerPipe[].
 *
 * @param  *base   pointer to main data structure
 */
void
installSigHandler (struct serverdata *base)
{
	struct sigaction sa = { {cbSigHandlerTX}, {{0}}, SA_RESTART, 0 };
	InputSource *src;
	int n = 0;
	
	if ((pipe(SigHandlerPipe)) == 0) {
		src = addInputSource (SigHandlerPipe[0], cbSigHandlerRD, base, FALSE);

		if (!sigaction (SIGINT,  &sa, NULL)) n++;
		if (!sigaction (SIGTERM, &sa, NULL)) n++;
		if (!sigaction (SIGHUP,  &sa, NULL)) n++;
		if (!sigaction (SIGUSR1, &sa, NULL)) n++;
		
		if (n == 0) {
			g_source_remove (src->watch);
			close (SigHandlerPipe[0]);
			close (SigHandlerPipe[1]);
		} else if (n < 4)
			print_msg (PBB_WARN, _("Not all signal handlers could be installed.\n"));
	} else
		print_msg (PBB_WARN, _("Can't install any signal handler\n"));
	return;
}

/**
 * @brief  Set up IPC server interface
 *
 * Clients connect to the server through a System V IPC interface.
 * This interface is a message port interface wich allows different
 * processes to communicate.
 *
 * This function opens the server port and installs a timer handler
 * to handle requests from clients.
 *
 * The System V IPC interface has no own authentification system.
 * To prevent unauthorized users to send IPC requests to pbbuttonsd,
 * the access could be limited to a priviledged user. This function
 * reads the user name from the global configuration file and sets
 * up a user filter for this user in the IPC backend.
 *
 * The IPC filter restricts write access to IPC but still allows
 * reading of IPC messages.
 */
int
ipc_init_stub ()
{
	int err;
	char *username;
	struct passwd *pwddata = NULL;

	err = ipc_init (NULL, LIBMODE_SERVER, 0);
	switch (err) {
	   case E_MSGPORT:
		print_msg (PBB_ERR, _("Can't create message port for server: %s.\n"), strerror(errno));
		break;
	   case E_TWICE:
		print_msg (PBB_ERR, _("Server is already running. Sorry, only one instance allowed.\n"));
		break;
	   case E_OLDPORT:
		print_msg (PBB_WARN, _("Orphaned server port found and removed. All running clients have to be restarted.\n"));
		err = 0;
	   case 0:
		if (config_has_key ("SYSTEM", "userallowed") == TRUE) {
			username = config_get_string ("SYSTEM", "userallowed", "paranoid");
			if ((pwddata = getpwnam(username)))
				ipc_filteruser(pwddata->pw_uid);
			else
				ipc_filterall();      /* switch off IPC-receive */
			g_free(username);
		}
		break;
	}

	register_function (T100QUEUE, ipc_handler);  /* input via IPC messages */
	return err;
}
